/*!
 * Foresight
 *
 * @author James Ooi <james.ooi@forefront.com.my>
 * @license MIT
 * @copyright 2018 (c) FOREFRONT International Sdn Bhd
 */
import 'intersection-observer';
/**
 * Available options for configuring Foresight.
 */
interface ForesightConfig {
    /** Defer tracking initialisation. */
    defer?: boolean;
    /** Configure the intersection observer. */
    observerOptions?: IntersectionObserverInit;
    /** Treat clicks as an interactive event. Defaults to true. */
    clicksAreInteractions?: boolean;
    /** Treat views as an interactive event. Defaults to false. */
    viewsAreInteractions?: boolean;
}
/**
 * Foresight is an analytics library that allows for declarative event tracking
 * in your websites.
 * @class
 */
declare class Foresight {
    /**
     * Default Options
     * @static
     */
    static defaultOptions: Partial<ForesightConfig>;
    /**
     * Stores the options of the current Foresight instance.
     * @public
     */
    options: ForesightConfig;
    /**
     * Stores a mapping of elements with is respective functions to de-register
     * listeners.
     * @private
     */
    private _untrackFns;
    /**
     * Stores an instance of an IntersectionObserver.
     * @private
     */
    private _observer;
    /**
     * @constructor
     */
    constructor(config: ForesightConfig);
    /**
     * Start event tracking for all DOM elements with event tracking attributes.
     */
    start(root?: Element): void;
    /**
     * Enables event tracking for a DOM element with event tracking attribute.
     */
    track(element: Element): void;
    /**
     * Disable event tracking for a DOM element.
     */
    untrack(element: Element): void;
    /**
     * Parse an event string and returns a `EventData` object.
     * @private
     */
    private _parseEventString;
    /**
     * Registers click listeners that triggers an analytics event when the element
     * is clicked or middle clicked.
     *
     * @returns Returns a function to remove the event listener.
     * @private
     */
    private _trackClicks;
    /**
     * Registers a view observer that triggers an analytics event when the element
     * is in view.
     *
     * @returns Returns a function that disconnects the view observer.
     * @private
     */
    private _trackViews;
    /**
     * Handles a click event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedClick;
    /**
     * Handles a view event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedView;
}
export = Foresight;
