/**!
 * Foresight
 *
 * @author James Ooi <james.ooi@forefront.com.my>
 * @license MIT
 * @copyright 2018 (c) FOREFRONT International Sdn Bhd
 */
import 'intersection-observer';
/**
 * Available options for configuring Foresight.
 */
interface ForesightConfig {
    /** Defer tracking initialisation. */
    defer?: boolean;
    /** Configure the intersection observer. */
    observerOptions?: IntersectionObserverInit;
    /** Treat clicks as an interactive event. Defaults to `true`. */
    clicksAreInteractions?: boolean;
    /** Treat views as an interactive event. Defaults to `false`. */
    viewsAreInteractions?: boolean;
    /** Override the default send event function. */
    sendEventFn?: (data: EventData) => any;
}
/**
 * Represents an event data.
 */
interface EventData {
    category: string;
    action: string;
    label: string;
    interaction: boolean;
    value?: string | number | null | undefined;
    metrics?: {
        [key: string]: any;
    };
    dimensions?: {
        [key: string]: any;
    };
}
/**
 * Foresight is an analytics library that allows for declarative event tracking.
 */
declare class Foresight {
    static defaultOptions: Partial<ForesightConfig>;
    /**
     * Stores the options of the current Foresight instance.
     * @public
     */
    options: ForesightConfig;
    /**
     * Stores a mapping of elements with its respective functions to de-register
     * the listeners.
     * @private
     */
    private _untrackFns;
    /**
     * Stores an instance of an IntersectionObserver.
     * @private
     */
    private _observer;
    /**
     * Store the current maximum page scroll amount.
     * @private
     */
    private _maxScrollY;
    /**
     * Indicates that the scroll tracker has already been initialised
     * @private
     */
    private _scrollTrackerInitialised;
    /**
     * @constructor
     */
    constructor(config?: ForesightConfig);
    /**
     * Start tracking all elements with tracking attributes for events.
     * @param {Element} root The container to scan for elements. Defaults to `body`.
     */
    start(root?: Element): void;
    /**
     * Start tracking an element for events.
     */
    track(element: Element): void;
    /**
     * Stop tracking an element for events.
     */
    untrack(element: Element): void;
    /**
     * Manually send an analytics event data.
     */
    send(data: EventData): any;
    /**
     * Parse an event string and returns a `EventData` object.
     * @private
     */
    private _parseEventString;
    /**
     * Registers click listeners that triggers an analytics event when the element
     * is clicked or middle clicked.
     *
     * @returns Returns a function to remove the event listener.
     * @private
     */
    private _trackClicks;
    /**
     * Registers a view observer that triggers an analytics event when the element
     * is in view.
     *
     * @returns Returns a function that disconnects the view observer.
     * @private
     */
    private _trackViews;
    /**
     * Handles a click event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedClick;
    /**
     * Handles a view event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedView;
    /**
     * Handles a scroll event
     */
    private _onScroll;
    private _onUnload;
}
export = Foresight;
