/*!
 * Foresight
 *
 * @author James Ooi <james.ooi@forefront.com.my>
 * @license MIT
 * @copyright 2018 (c) FOREFRONT International Sdn Bhd
 */
import 'intersection-observer';
/**
 * Available options for configuring Foresight.
 */
interface ForesightConfig {
    /** Defer tracking initialisation. */
    defer?: boolean;
    /** Configure the intersection observer. */
    observerOptions?: IntersectionObserverInit;
    /** Treat clicks as an interactive event. Defaults to `true`. */
    clicksAreInteractions?: boolean;
    /** Treat views as an interactive event. Defaults to `false`. */
    viewsAreInteractions?: boolean;
    /** Override the default send event function. */
    sendEventFn?: (data: EventData) => any;
}
/**
 * Represents a particular event's data.
 */
interface EventData {
    category: string;
    action: string;
    label: string;
    interaction: boolean;
}
/**
 * Foresight is an analytics library that allows for declarative event tracking.
 * @class
 */
declare class Foresight {
    /**
     * Default Options
     * @static
     */
    static defaultOptions: Partial<ForesightConfig>;
    /**
     * Stores the options of the current Foresight instance.
     * @public
     */
    options: ForesightConfig;
    /**
     * Stores a mapping of elements with is respective functions to de-register
     * listeners.
     * @private
     */
    private _untrackFns;
    /**
     * Stores an instance of an IntersectionObserver.
     * @private
     */
    private _observer;
    /**
     * @constructor
     */
    constructor(config?: ForesightConfig);
    /**
     * Start tracking all elements with tracking attributes for events.
     * @param {Element} root The container to scan for elements. Defaults to `body`.
     */
    start(root?: Element): void;
    /**
     * Start tracking an element for events.
     */
    track(element: Element): void;
    /**
     * Stop tracking an element for events.
     */
    untrack(element: Element): void;
    /**
     * Manually send an analytics event data.
     */
    send(data: EventData): any;
    /**
     * Parse an event string and returns a `EventData` object.
     * @private
     */
    private _parseEventString;
    /**
     * Registers click listeners that triggers an analytics event when the element
     * is clicked or middle clicked.
     *
     * @returns Returns a function to remove the event listener.
     * @private
     */
    private _trackClicks;
    /**
     * Registers a view observer that triggers an analytics event when the element
     * is in view.
     *
     * @returns Returns a function that disconnects the view observer.
     * @private
     */
    private _trackViews;
    /**
     * Handles a click event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedClick;
    /**
     * Handles a view event on an element that is being tracked by Foresight.
     * @private
     */
    private _onTrackedView;
}
export = Foresight;
