/**!
 * Foresight
 *
 * @author James Ooi <james.ooi@forefront.com.my>
 * @license MIT
 * @copyright 2018 (c) FOREFRONT International Sdn Bhd
 */

import 'intersection-observer';
import * as Utils from './utils';


/**
 * Available options for configuring Foresight.
 */
interface ForesightConfig {
  /** Defer tracking initialisation. */
  defer?: boolean

  /** Configure the intersection observer. */
  observerOptions?: IntersectionObserverInit

  /** Treat clicks as an interactive event. Defaults to `true`. */
  clicksAreInteractions?: boolean

  /** Treat views as an interactive event. Defaults to `false`. */
  viewsAreInteractions?: boolean

  /** Override the default send event function. */
  sendEventFn?: (data: EventData) => any
}


/**
 * Represents an event data.
 */
interface EventData {
  category: string
  action: string
  label: string
  interaction: boolean
  value?: string | number | null | undefined,
  metrics?: { [key: string]: any }
  dimensions?: { [key: string]: any }
}


/**
 * Foresight is an analytics library that allows for declarative event tracking.
 */
class Foresight {

  static defaultOptions: Partial<ForesightConfig> = {
    defer: false,
    observerOptions: {},
    clicksAreInteractions: true,
    viewsAreInteractions: false,
    sendEventFn: null,
  }

  /**
   * Stores the options of the current Foresight instance.
   * @public
   */
  options: ForesightConfig;

  /**
   * Stores a mapping of elements with its respective functions to de-register
   * the listeners.
   * @private
   */
  private _untrackFns: Map<Element, { click: Function, view: Function }> = new Map();

  /**
   * Stores an instance of an IntersectionObserver.
   * @private
   */
  private _observer: IntersectionObserver = null;

  /**
   * Store the current maximum page scroll amount.
   * @private
   */
  private _maxScrollY: number = 0;

  /**
   * Indicates that the scroll tracker has already been initialised
   * @private
   */
  private _scrollTrackerInitialised: boolean = false;

  /**
   * @constructor
   */
  constructor(config: ForesightConfig = {}) {
    this.options = { ...Foresight.defaultOptions, ...config };

    // Initialise IntersectionObserver
    this._observer = new IntersectionObserver((entries, observer) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          this._onTrackedView(entry.target, observer);
          observer.unobserve(entry.target);
        }
      });
    }, this.options.observerOptions);

    // Start tracking
    if (!this.options.defer) {
      const domContentLoaded = document.readyState === 'complete' || document.readyState === 'interactive';
      if (domContentLoaded) {
        this.start()
      } else {
        window.addEventListener('DOMContentLoaded', () => this.start());
      }
    }
  }

  /**
   * Start tracking all elements with tracking attributes for events.
   * @param {Element} root The container to scan for elements. Defaults to `body`.
   */
  start(root: Element = document.body) {
    Utils
      .toArray<Element>(root.querySelectorAll('[data-track], [data-track-view]'))
      .map(element => this.track(element));

    if (!this._scrollTrackerInitialised) {
      const maxScrollListener = Utils.debounce(() => this._onScroll());
      window.addEventListener('scroll', maxScrollListener, { passive: true });
      this._onScroll(); // call once
      this._scrollTrackerInitialised = true;
    }

    window.addEventListener('beforeunload', (e) => this._onUnload(e));
  }

  /**
   * Start tracking an element for events.
   */
  track(element: Element) {
    if (!this._untrackFns.has(element)) {
      this._untrackFns.set(element, { click: null, view: null });
    }

    const untrackFn = this._untrackFns.get(element);

    // Track clicks
    if (element.getAttribute('data-track') !== null) {
      if (untrackFn.click == null) {
        untrackFn.click = this._trackClicks(element);
      }
    }

    // Track views
    if (element.getAttribute('data-track-view') !== null) {
      if (untrackFn.view == null) {
        untrackFn.view = this._trackViews(element);
      }
    }

    this._untrackFns.set(element, untrackFn);
  }

  /**
   * Stop tracking an element for events.
   */
  untrack(element: Element) {
    const untrackFn = this._untrackFns.get(element);

    if (untrackFn === undefined) {
      return;
    }

    if (untrackFn.click !== null) {
      untrackFn.click();
    }

    if (untrackFn.view !== null) {
      untrackFn.view();
    }

    this._untrackFns.delete(element);
  }

  /**
   * Manually send an analytics event data.
   */
  send(data: EventData) {
    if (this.options.sendEventFn !== null) {
      return this.options.sendEventFn(data);
    }

    if (typeof window['gtm'] === 'function') {
      return window['gtm']({
        'event': data.action,
        'event_label': data.label,
        'event_category': data.category,
        'non_interaction': !data.interaction,
        ...data.metrics,
        ...data.dimensions
      });
    }

    if (typeof window['gtag'] === 'function') {
      return window['gtag']('event', data.action, {
        ...data.metrics,
        ...data.dimensions,
        ...(data.label ? { label: data.label } : {}),
        ...(data.category ? { category: data.category } : {}),
        'non_interaction': !data.interaction
      });
    }

    if (typeof window['ga'] === 'function') {
      return window['ga']('send', {
        hitType: 'event',
        eventCategory: data.category,
        eventAction: data.action,
        eventLabel: data.label,
        nonInteraction: !data.interaction,
      }, { ...data.metrics, ...data.dimensions });
    }

    throw 'No available analytics backend found. Has Google Analytics been loaded yet?';
  }

  /**
   * Parse an event string and returns a `EventData` object.
   * @private
   */
  private _parseEventString(eventString: string): Partial<EventData> {
    const split = eventString.split(';').map(s => s.trim());
    let [ category, action, label ] = split;

    // If only one argument is provided, then the argument is the action
    if (split.length === 1) {
      action = category;
      category = undefined;
    }

    return { category, action, label, interaction: true }
  }

  /**
   * Registers click listeners that triggers an analytics event when the element
   * is clicked or middle clicked.
   *
   * @returns Returns a function to remove the event listener.
   * @private
   */
  private _trackClicks(element: Element): Function {
    // Define listen function
    const listener = (e) => this._onTrackedClick(element, e);

    element.addEventListener('click', listener);
    element.addEventListener('auxclick', listener);

    return () => {
      element.removeEventListener('click', listener);
      element.removeEventListener('auxclick', listener);
    };
  }


  /**
   * Registers a view observer that triggers an analytics event when the element
   * is in view.
   *
   * @returns Returns a function that disconnects the view observer.
   * @private
   */
  private _trackViews(element: Element): Function {
    this._observer.observe(element);

    return () => {
      this._observer.unobserve(element);
    };
  }


  /**
   * Handles a click event on an element that is being tracked by Foresight.
   * @private
   */
  private _onTrackedClick(element: Element, event: Event) {
    const s = element.getAttribute('data-track');
    const data = this._parseEventString(s);

    // Get custom metrics
    data.metrics = Utils
      .toArray<Attr>(element.attributes)
      .filter(attr => attr.name.startsWith('data-track:metrics'))
      .reduce((acc, attr) => {
        const key = attr.name.replace(/data-track:metrics:/, '');
        const value = attr.value === "" ? 1 : attr.value;
        acc[key] = value;
        return acc;
      }, {});

    // Get custom dimensions
    data.dimensions = Utils
      .toArray<Attr>(element.attributes)
      .filter(attr => attr.name.startsWith('data-track:dimensions'))
      .reduce((acc, attr) => {
        if (attr.value === "") {
          console.warn(`No dimension value provided for ${attr.name}`);
          return acc;
        }
        const key = attr.name.replace(/data-track:dimensions:/, '');
        const value = attr.value;
        acc[key] = value;
        return acc;
      }, {});

    // Get interaction flag
    data.interaction = this.options.clicksAreInteractions;
    if (element.getAttribute('data-track:non-interaction') !== null) {
      data.interaction = false;
    }

    this.send(<EventData> data);

    if (element.hasAttribute('data-track:page-view')) {
      const pagePath = element.getAttribute('data-track:page-view') || window.location.pathname;
      this.sendPageView(pagePath);
    }
  }

  /**
   * Handles a view event on an element that is being tracked by Foresight.
   * @private
   */
  private _onTrackedView(element: Element, observer) {
    const s = element.getAttribute('data-track-view');
    const data = this._parseEventString(s);

    // Get custom metrics
    data.metrics = Utils
      .toArray<Attr>(element.attributes)
      .filter(attr => attr.name.startsWith('data-track-view:metrics'))
      .reduce((acc, attr) => {
        const key = attr.name.replace(/data-track-view:metrics:/, '');
        const value = attr.value === "" ? 1 : attr.value;
        acc[key] = value;
        return acc;
      }, {});

    // Get custom dimensions
    data.dimensions = Utils
      .toArray<Attr>(element.attributes)
      .filter(attr => attr.name.startsWith('data-track-view:dimensions'))
      .reduce((acc, attr) => {
        if (attr.value === "") {
          console.warn(`No dimension value provided for ${attr.name}`);
          return acc;
        }
        const key = attr.name.replace(/data-track-view:dimensions:/, '');
        const value = attr.value;
        acc[key] = value;
        return acc;
      }, {});

    // Get interaction flag
    data.interaction = this.options.viewsAreInteractions;
    if (element.getAttribute('data-track-view:interaction') !== null) {
      data.interaction = true;
    }

    this.send(<EventData> data);
  }

  /**
   * Handles a scroll event
   */
  private _onScroll(root = document.documentElement) {
    const vh = Math.max(root.clientHeight, window.innerHeight || 0);
    const scrollY = root.scrollTop + vh;
    if (scrollY > this._maxScrollY) {
      this._maxScrollY = scrollY;
    }
  }

  private _onUnload(e: Event) {
    const scrollData: EventData = {
      category: 'general',
      action: 'scroll y',
      label: window.location.pathname,
      interaction: false,
      value: Math.floor((this._maxScrollY / document.body.clientHeight) * 100)
    }
    this.send(scrollData);
  }

  /**
   * Send a page view event manually.
   * @param {string} pagePath The page path to send. Defaults to current location.
   */
  sendPageView(pagePath: string = window.location.pathname) {
    if (typeof window['gtag'] === 'function') {
      window['gtag']('event', 'page_view', {
        page_path: pagePath
      });
    }
  }
}

export = Foresight;
