/**
 * Foresight
 *
 * @author James Ooi <james.ooi@forefront.com.my>
 * @license MIT
 * @copyright 2018 (c) FOREFRONT International Sdn Bhd
 */

import 'intersection-observer';
import * as Utils from './utils';

declare var gtag: (action: string, ...args: any[]) => any;


/**
 * Available options for configuring Foresight.
 */
interface ForesightConfig {
  observerOptions?: IntersectionObserverInit
  nonInteractionClicks?: boolean
  nonInteractionViews?: boolean
}


/**
 * Represents a particular event's data.
 */
interface EventData {
  category: string
  action: string
  label: string
  interaction: boolean
}


/**
 * Foresight is an analytics library that allows for declarative event tracking
 * in your websites.
 * @class
 */
class Foresight {

  /**
   * Default Options
   * @static
   */
  static defaultOptions: Partial<ForesightConfig> = {
    nonInteractionClicks: false,
    nonInteractionViews: true,
    observerOptions: {},
  }

  /**
   * Stores the options of the current Foresight instance.
   * @public
   */
  options: ForesightConfig;

  /**
   * Stores a mapping of elements with is respective functions to de-register
   * listeners.
   * @private
   */
  private _untrackFns: Map<Element, { click: Function, view: Function }> = new Map();

  /**
   * Stores an instance of an IntersectionObserver.
   * @private
   */
  private _observer: IntersectionObserver = null;

  /**
   * @constructor
   */
  constructor(config: ForesightConfig) {
    this.options = { ...Foresight.defaultOptions, ...config };

    // Initialise IntersectionObserver
    this._observer = new IntersectionObserver((entries, observer) => {
      entries.forEach(entry => {
        if (entry.isIntersecting) {
          this._onTrackedView(entry.target, observer);
          observer.unobserve(entry.target);
        }
      });
    }, this.options.observerOptions);
  }

  /**
   * Start event tracking for all DOM elements with event tracking attributes.
   */
  start(root: Element = document.body) {
    if (gtag === undefined) {
      throw `The 'gtag' function is undefined. Has Google Analytics been loaded yet?`;
    }

    Utils
      .toArray<Element>(root.querySelectorAll('[data-track], [data-track-view]'))
      .map(element => this.track(element));
  }

  /**
   * Enables event tracking for a DOM element with event tracking attribute.
   */
  track(element: Element) {
    if (!this._untrackFns.has(element)) {
      this._untrackFns.set(element, { click: null, view: null });
    }

    const untrackFn = this._untrackFns.get(element);

    // Track clicks
    if (element.getAttribute('data-track') !== null) {
      if (untrackFn.click == null) {
        untrackFn.click = this._trackClicks(element);
      }
    }

    // Track views
    if (element.getAttribute('data-track-view') !== null) {
      if (untrackFn.view == null) {
        untrackFn.view = this._trackViews(element);
      }
    }

    this._untrackFns.set(element, untrackFn);
  }

  /**
   * Disable event tracking for a DOM element.
   */
  untrack(element: Element) {
    const untrackFn = this._untrackFns.get(element);

    if (untrackFn === undefined) {
      return;
    }

    if (untrackFn.click !== null) {
      untrackFn.click();
    }

    if (untrackFn.view !== null) {
      untrackFn.view();
    }

    this._untrackFns.delete(element);
  }

  /**
   * Parse an event string and returns a `EventData` object.
   * @private
   */
  private _parseEventString(eventString: string): EventData {
    const split = eventString.split(';');
    let [ category, action, label ] = split;

    // If only one argument is provided, then the argument is the action
    if (split.length === 1) {
      action = category;
      category = undefined;
    }

    return { category, action, label, interaction: true }
  }

  /**
   * Registers click listeners that triggers an analytics event when the element
   * is clicked or middle clicked.
   *
   * @returns Returns a function to remove the event listener.
   * @private
   */
  private _trackClicks(element: Element): Function {
    // Define listen fucntion
    const listener = (e) => {
      this._onTrackedClick(element, e);
    };

    element.addEventListener('click', listener);
    element.addEventListener('auxclick', listener);

    return () => {
      element.removeEventListener('click', listener);
      element.removeEventListener('auxclick', listener);
    };
  }


  /**
   * Registers a view observer that triggers an analytics event when the element
   * is in view.
   *
   * @returns Returns a function that disconnects the view observer.
   * @private
   */
  private _trackViews(element: Element): Function {
    this._observer.observe(element);

    return () => {
      this._observer.unobserve(element);
    };
  }


  /**
   * Handles a click event on an element that is being tracked by Foresight.
   * @private
   */
  private _onTrackedClick(element: Element, event: Event) {
    const s = element.getAttribute('data-track');
    const data = this._parseEventString(s);

    data.interaction = !this.options.nonInteractionClicks;
    if (element.getAttribute('data-track:non-interaction') !== null) {
      data.interaction = false;
    }

    gtag('event', data.action, {
      'event_label': data.label,
      'event_category': data.category,
      'non_interaction': !data.interaction,
    });
  }

  /**
   * Handles a view event on an element that is being tracked by Foresight.
   * @private
   */
  private _onTrackedView(element: Element, observer) {
    const s = element.getAttribute('data-track-view');
    const data = this._parseEventString(s);

    data.interaction = !this.options.nonInteractionViews;
    if (element.getAttribute('data-track-view:interaction') !== null) {
      data.interaction = true;
    }

    gtag('event', data.action, {
      'event_label': data.label,
      'event_category': data.category,
      'non_interaction': !data.interaction,
    });
  }
}

export = Foresight;
